/*
 *  Author: Blanc Nicolas
 *  
 *
 *
 *  Mock Routing-Protocol Model.
 * ----------------------------
 * 
 * Involved agents: machines, routers.
 * 
 * Machines send packets to other machines.
 * Machines recognize the following actions:
 *   - EMIT: randomly emit a packet to a random destination
 *   - LISTEN: listen for the arrival of packets
 *
 * Routers listen for the arrival of packets and possibly
 * forward them using two outgoing lines. Routers perform the following
 * actions:
 *  - FORWARD(dest, line): forward a packet on its way to 'dest'
 *      through line 0/1.
 *  - BROADCAST: forward a packet through line 0 and line 1.
 *  - DROP: drop a packet if its time to live has expired.
 *
 */

#include <systemc.h>
#include <tlm.h>
#include <iostream>
#include <vector>
#include <stdlib.h>
#include "tlm_fifo_ports.h"

using namespace tlm;

//---------------------------------------------------------------
// Action Macros
//---------------------------------------------------------------
#define ROUTER_ACTION_FORWARD(agent, address, line)  agent.action_table.push_back(new Routert::ForwardActiont(address, line, #agent "::Forward(Address " #address ", Line " #line ")\n"));
#define ROUTER_ACTION_BROADCAST(agent) agent.action_table.push_back(new Routert::BroadCastActiont(#agent "::BroadCast\n"));
#define ROUTER_ACTION_DROP(agent) agent.action_table.push_back(new Routert::DropActiont(#agent "::Drop\n"));

#define MACHINE_ACTION_LISTEN(agent) agent.action_table.push_back(new Machinet::ListenActiont(#agent "::Listen\n"));
#define MACHINE_ACTION_EMIT(agent) agent.action_table.push_back(new Machinet::EmitActiont(#agent "::Emit\n"));

//---------------------------------------------------------------
// Packett
//---------------------------------------------------------------

typedef sc_uint<3> addresst;

struct Packett
{
  addresst address;      // destination address
  sc_uint<8> ttl;        // time-to-live of the packet
  sc_uint<32>  data;     // data transmitted
};


//---------------------------------------------------------------
//
// Routert
//
//---------------------------------------------------------------

SC_MODULE(Routert)
{
  //---------------------------------------------------------------
  // Forward Class Declarations
  //---------------------------------------------------------------

  class ActionManagert;     // Base class for all managers
  class GuardManagert;      // Class of the guard manager
  class TransitionManagert; // Class of the transition manager

  class Actiont;            // Base class for all actions
  class ForwardActiont;     // Action class for forwarding a packet
  class BroadCastActiont;   // Action class for broadcasting a packet
  class DropActiont;        // Action class for dropping a packet


  //---------------------------------------------------------------
  // Port Declarations
  //---------------------------------------------------------------

  sc_in<bool> clk;                    // Synchronization clock

  tlm_fifo_nb_in<Packett>  line_in;   // Input channel of the router
  tlm_fifo_nb_out<Packett> line_out0; // First output channel of the router
  tlm_fifo_nb_out<Packett> line_out1; // Second output channel of the router


  //---------------------------------------------------------------
  // Module's Constructor
  //---------------------------------------------------------------

  SC_CTOR(Routert):gm(line_in),tm(line_in, line_out0, line_out1)
  {
    SC_METHOD(process);
    sensitive << clk;
  }


  //---------------------------------------------------------------
  // Module's Method Process
  //---------------------------------------------------------------

  void process()
  {
    // Execute the first enabled action
    for(int i = 0; i < action_table.size(); i++)
    {
      const Actiont& a = *(action_table[i]);
      if(gm.isEnabled(a))
      {
	std::cout << a.getId();// << endl;
        tm.perform(a);
        return;
      }
    }
  }


  //---------------------------------------------------------------
  // ActionManagert
  //---------------------------------------------------------------

  class ActionManagert
  {
    public:
    virtual void caseForwardActiont(const ForwardActiont&) = 0;
    virtual void caseBroadCastActiont(const BroadCastActiont&) = 0;
    virtual void caseDropActiont(const DropActiont&) = 0;
  };


  //---------------------------------------------------------------
  // Actiont
  //---------------------------------------------------------------

  class Actiont
  {
    public:
    Actiont(const char* id): id(id){}

    virtual void apply(ActionManagert&) const = 0;

    const char* getId() const
    { return id; }

    private:
    Actiont(const Actiont&);             // disabled
    Actiont& operator=( const Actiont&); // disabled

    const char* id;
  };


  //---------------------------------------------------------------
  // ForwardActiont
  //---------------------------------------------------------------

  class ForwardActiont: public Actiont
  {
    public:
    ForwardActiont(addresst address,
                  bool line,
                  const char* id):
    Actiont(id), address(address), line(line){}

    virtual void apply(ActionManagert& am) const
    { am.caseForwardActiont(*this); }

    addresst get_address() const
    { return address; }

    bool get_line() const
    { return line; }

    private:
    ForwardActiont(const ForwardActiont&);             // disabled
    ForwardActiont& operator=( const ForwardActiont&); // disabled

    addresst address;
    bool line;
  };


  //---------------------------------------------------------------
  // BroadCastActiont
  //---------------------------------------------------------------

  class BroadCastActiont: public Actiont
  {
    public:
    BroadCastActiont(const char* id): Actiont(id){}
    virtual void apply(ActionManagert& am) const
    { am.caseBroadCastActiont(*this); }

    private:
    BroadCastActiont(const BroadCastActiont&);             // disabled
    BroadCastActiont& operator= (const BroadCastActiont&); // disabled
  };


  //---------------------------------------------------------------
  // DropActiont
  //---------------------------------------------------------------

  class DropActiont: public Actiont
  {
    public:
    DropActiont(const char* id): Actiont(id){}

    virtual void apply(ActionManagert& am) const
    { am.caseDropActiont(*this); }

    private:
    DropActiont(const DropActiont&);              // disabled
    DropActiont& operator= (const DropActiont&);  // disabled

  };


  //---------------------------------------------------------------
  // GuardManager
  //---------------------------------------------------------------

  class GuardManagert: private ActionManagert
  {
    public:
    GuardManagert(const tlm_fifo_nb_in<Packett>&  line_in):
    line_in(line_in) {}

    bool isEnabled(const Actiont& a)
    {
      is_action_enabled = false;
      a.apply(*this);
      return is_action_enabled;

    }

    private:

    virtual void caseForwardActiont(const ForwardActiont& a)
    {
      Packett p;
      is_action_enabled = line_in.nb_peek(p) &&
      	p.ttl > 0 &&
      	p.address == a.get_address();
    }

    virtual void caseBroadCastActiont(const BroadCastActiont& a)
    {
      Packett p;
      is_action_enabled = line_in.nb_peek(p) && p.ttl > 0;
    }

    virtual void caseDropActiont(const DropActiont& a)
    {
      Packett p;
      is_action_enabled = line_in.nb_peek(p);
    }

    const tlm_fifo_nb_in<Packett>&  line_in;

    bool is_action_enabled;

  }; // GuardManagert


  //---------------------------------------------------------------
  // TransitionManagert
  //---------------------------------------------------------------

  class TransitionManagert: private ActionManagert
  {
    public:
    // Transitin Manager' Constructor
    TransitionManagert(tlm_fifo_nb_in<Packett>&  line_in,
                       tlm_fifo_nb_out<Packett>&  line_out0,
                       tlm_fifo_nb_out<Packett>&  line_out1
                       ):
    line_in(line_in), line_out0(line_out0), line_out1(line_out1) {}

    // Perform the action given as parameter
    void perform(const Actiont& a)
    { a.apply(*this); }

    private:

    virtual void caseForwardActiont(const ForwardActiont& a)
    {
      Packett p;
      bool outcome = line_in.nb_get(p);
      assert(outcome == true);

      p.ttl--;

      if(a.get_line() == true)
        line_out1.nb_put(p);
      else
        line_out0.nb_put(p);
    };

    virtual void caseBroadCastActiont(const BroadCastActiont& a)
    {
      Packett p;
      bool outcome = line_in.nb_get(p);
      assert(outcome == true);

      p.ttl--;

      line_out0.nb_put(p);
      line_out1.nb_put(p);
    };

    virtual void caseDropActiont(const DropActiont& a)
    {
      Packett p;
      line_in.nb_get(p);
    };

    tlm_fifo_nb_in<Packett>&  line_in;
    tlm_fifo_nb_out<Packett>&  line_out0;
    tlm_fifo_nb_out<Packett>&  line_out1;

  }; // TransitionManagert


  GuardManagert gm;                         // Guard Manager
  TransitionManagert tm;                    // Transition Manager

  std::vector<const Actiont*> action_table;      // Action Table

}; // Routert




//---------------------------------------------------------------
//
// Machinet
//
//---------------------------------------------------------------

SC_MODULE(Machinet)
{
  //---------------------------------------------------------------
  // Forward Class Declarations
  //---------------------------------------------------------------

  class ActionManagert;     // Base class for all managers
  class GuardManagert;      // Class of the guard manager
  class TransitionManagert; // Class of the transition manager

  class Actiont;            // Base class for all actions
  class ListenActiont;      // Action class for listening to the input line
  class EmitActiont;        // Action class for emitting a packet


  //---------------------------------------------------------------
  // Port Declarations
  //---------------------------------------------------------------

  sc_in<bool> clk;                    // Synchronization clock
  tlm_fifo_nb_in<Packett>  line_in;   // Input channel
  tlm_fifo_nb_out<Packett> line_out;  // Output channel


  //---------------------------------------------------------------
  // Module's Constructor
  //---------------------------------------------------------------

  SC_CTOR(Machinet):gm(line_in),tm(line_in, line_out)
  {
    SC_METHOD(process);
    sensitive << clk;

  }


  //---------------------------------------------------------------
  // Module's Method Process
  //---------------------------------------------------------------

  void process()
  {
    for(int i = 0; i < action_table.size(); i++)
    {
      const Actiont& a = *(action_table[i]);
      if(gm.isEnabled(a))
      {
	      std::cout << a.getId();// << endl;
        tm.perform(a);
        return;
      }
    }
  }


  //---------------------------------------------------------------
  // ActionManagert
  //---------------------------------------------------------------

  class ActionManagert
  {
    public:
    virtual void caseListenActiont(const ListenActiont&) = 0;
    virtual void caseEmitActiont(const EmitActiont&) = 0;
  };


  //---------------------------------------------------------------
  // Actiont
  //---------------------------------------------------------------

  class Actiont
  {
    public:
    Actiont(const char* id): id(id){}

    virtual void apply(ActionManagert&) const = 0;

    const char* getId() const
    { return id; }

    private:
    Actiont(const Actiont&);              // disabled
    Actiont& operator=( const Actiont&);  // disabled

    const char* id;
  };


  //---------------------------------------------------------------
  // ListenActiont
  //---------------------------------------------------------------

  class ListenActiont: public Actiont
  {
    public:
    ListenActiont(const char* id): Actiont(id){}

    virtual void apply(ActionManagert& am) const
    { am.caseListenActiont(*this); }

    private:
    ListenActiont(const ListenActiont&);             // disabled
    ListenActiont& operator= (const ListenActiont&); // disabled

  };


  //---------------------------------------------------------------
  // EmitActiont
  //---------------------------------------------------------------

  class EmitActiont: public Actiont
  {
    public:
    EmitActiont(const char* id): Actiont(id){}

    virtual void apply(ActionManagert& am) const
    { am.caseEmitActiont(*this); }

    private:
    EmitActiont(const EmitActiont&);             // disabled
    EmitActiont& operator= (const EmitActiont&); // disabled

  };


  //---------------------------------------------------------------
  // GuardManagert
  //---------------------------------------------------------------

  class GuardManagert: private ActionManagert
  {
    public:
    GuardManagert(const tlm_fifo_nb_in<Packett>&  line_in):
    line_in(line_in) {}

    bool isEnabled(const Actiont& a)
    {
      is_action_enabled = false;
      a.apply(*this);
      return is_action_enabled;
    }

    private:

    virtual void caseListenActiont(const ListenActiont& a)
    {
      Packett p;
      is_action_enabled = line_in.nb_peek(p);
    }

    virtual void caseEmitActiont(const EmitActiont& a)
    {
      // randomly emit a packet
      is_action_enabled = (rand() <  (RAND_MAX/4));
    }

    const tlm_fifo_nb_in<Packett>&  line_in;

    bool is_action_enabled;

  }; // GuardManagert


  //---------------------------------------------------------------
  // TransitionManagert
  //---------------------------------------------------------------

  class TransitionManagert: private ActionManagert
  {
    public:
    TransitionManagert(tlm_fifo_nb_in<Packett>&  line_in,
                       tlm_fifo_nb_out<Packett>&  line_out
                       ):
    line_in(line_in), line_out(line_out) {}

    void perform(const Actiont& a)
    { a.apply(*this); }

    private:

    virtual void caseListenActiont(const ListenActiont& a)
    {
      Packett p;
      bool outcome = line_in.nb_get(p);
      assert(outcome == true);
    };

    virtual void caseEmitActiont(const EmitActiont& a)
    {
      // Send a packet to a random destination
      static sc_uint<32> counter;

      Packett p;
      p.address = rand();
      p.data = counter++;
      p.ttl = 3;

      line_out.nb_put(p);
    }

    tlm_fifo_nb_in<Packett>&   line_in;
    tlm_fifo_nb_out<Packett>&  line_out;

  }; // TransitionManagert


  GuardManagert gm;                         // Guard Manager
  TransitionManagert tm;                    // Transition Manager

  std::vector<const Actiont*> action_table; // Action Table

}; // Machinet




//---------------------------------------------------------------
//
// sc_main
//
//---------------------------------------------------------------

int sc_main(int argc, char* argv[])
{
  //
  // Module Declarations
  //

  Machinet m0("m0");
  Machinet m1("m1");
  Machinet m2("m2");
  Machinet m3("m3");

  Routert r0("r0");
  Routert r1("r1");
  Routert r2("r2");
  Routert r3("r3");


  //
  // Communication-channel Declarations
  //

  sc_clock clk("clk");  // Global synchornization clock

  tlm_fifo<Packett> fifo_r0_m0("fifo_r0_m0",16);
  tlm_fifo<Packett> fifo_r0_m1("fifo_r0_m1",16);
  tlm_fifo<Packett> fifo_r1_m2("fifo_r1_m2",16);
  tlm_fifo<Packett> fifo_r1_m3("fifo_r1_m3",16);
  tlm_fifo<Packett> fifo_r2_r0("fifo_r2_r0",16);
  tlm_fifo<Packett> fifo_r3_r1("fifo_r3_r1",16);
  tlm_fifo<Packett> fifo_r2_in("fifo_r2_in",16);
  tlm_fifo<Packett> fifo_r3_in("fifo_r3_in",16);


  //
  // Port Binding
  //

  m0.clk(clk);
  m0.line_in(fifo_r0_m0);
  m0.line_out(fifo_r2_in);

  m1.clk(clk);
  m1.line_in(fifo_r0_m1);
  m1.line_out(fifo_r2_in);

  m2.clk(clk);
  m2.line_in(fifo_r1_m2);
  m2.line_out(fifo_r3_in);

  m3.clk(clk);
  m3.line_in(fifo_r1_m3);
  m3.line_out(fifo_r3_in);

  r0.clk(clk);
  r0.line_in(fifo_r2_r0);
  r0.line_out0(fifo_r0_m0);
  r0.line_out1(fifo_r0_m1);

  r1.clk(clk);
  r1.line_in(fifo_r3_r1);
  r1.line_out0(fifo_r1_m2);
  r1.line_out1(fifo_r1_m3);

  r2.clk(clk);
  r2.line_in(fifo_r2_in);
  r2.line_out0(fifo_r2_r0);
  r2.line_out1(fifo_r3_in);

  r3.clk(clk);
  r3.line_in(fifo_r3_in);
  r3.line_out0(fifo_r3_r1);
  r3.line_out1(fifo_r2_in);


  //
  // Action Tables
  //

  MACHINE_ACTION_EMIT(m0);
  MACHINE_ACTION_LISTEN(m0);

  MACHINE_ACTION_EMIT(m1);
  MACHINE_ACTION_LISTEN(m1);

  MACHINE_ACTION_EMIT(m2);
  MACHINE_ACTION_LISTEN(m2);

  MACHINE_ACTION_EMIT(m3);
  MACHINE_ACTION_LISTEN(m3);

  ROUTER_ACTION_FORWARD(r0, 0, false);
  ROUTER_ACTION_FORWARD(r0, 1, true);
  ROUTER_ACTION_DROP(r0);

  ROUTER_ACTION_FORWARD(r1, 2, false);
  ROUTER_ACTION_FORWARD(r1, 3, true);
  ROUTER_ACTION_DROP(r1);

  ROUTER_ACTION_FORWARD(r2, 0, false);
  ROUTER_ACTION_FORWARD(r2, 1, false);
  ROUTER_ACTION_BROADCAST(r2);
  ROUTER_ACTION_DROP(r2);

  ROUTER_ACTION_FORWARD(r3, 2, false);
  ROUTER_ACTION_FORWARD(r3, 3, false);
  ROUTER_ACTION_BROADCAST(r3);
  ROUTER_ACTION_DROP(r3);

  sc_start(1, SC_US);
  return 0;
}
