/*******************************************************************\
 *
 *
 * Author: Blanc Nicolas
 *
 * 
\*******************************************************************/


#ifndef __UNSIGNEDBV_H
#define __UNSIGNEDBV_H


#include <limits.h>
#include <assert.h>
#include <iostream>
#include <stdio.h>

#define UNSIGNEDBV_SIZEOF_DATA(x)   (sizeof(x)/sizeof(unsigned int))

static const unsigned  UINT_BSIZE = sizeof(unsigned int)*8;
static const unsigned  LONG_LONG_UINT_BSIZE = sizeof(long long int)*8;

template <int W>
struct __signedbv;

template <int W>
struct __unsignedbv;

template<int W>
std::ostream& operator << ( std::ostream& os, const __unsignedbv<W>& bv);

template <int W>
struct __unsignedbv
{
  // For storing bits in little endian mode
  unsigned int data[((W-1)/UINT_BSIZE)+1];

  // Default construcor
  __unsignedbv(){};

  // Copy constructor
  __unsignedbv(const __unsignedbv<W>& bv)
  {
    memcpy(data,bv.data,sizeof(data));
  }

  // Constructor for integer type
  __unsignedbv(int v)
  {
    data[0] = v;
    for(unsigned i = 1; i < UNSIGNEDBV_SIZEOF_DATA(data); i++)
      data_at(i) = 0;
  }

  // Conversion constructor for type __unsignedbv
  template <int W2>
    explicit __unsignedbv(const __unsignedbv<W2>& bv);

  // Conversion constructor form _unsignedbv to __signedbv
  template <int W2>
    explicit __unsignedbv(const __signedbv<W2>& bv);

  // Constructor reading a string that represents
  // an integer in binary format
  explicit __unsignedbv(const char binstr [W+1])
  {
#ifdef __BV_DEBUG
    assert(binstr[W] == '\0');
#endif

    int index = W-1;
    for(unsigned j = 0; j < W/UINT_BSIZE; j++)
    {
      unsigned acc = 0;
      for(unsigned i = 0; i < UINT_BSIZE; i++)
      {
        acc >>= 1;
        acc += (binstr[index] == '0' ? 0: 1) << 31;
        index--;
      }
      data_at(j)=acc;
    }

    if(W%UINT_BSIZE != 0)
    {
      unsigned acc = 0;
      for(; index >= 0; index--)
      {
        acc >>=1;
        acc += (binstr[index] == '0' ? 0: 1) << 31;
      }
      acc >>= UINT_BSIZE-(W%UINT_BSIZE);
      data_at(UNSIGNEDBV_SIZEOF_DATA(data)-1) = acc;
    }
  }

  // Bit-extraction constructor

  template<int W2>
    __unsignedbv(const __unsignedbv<W2>& bv, int left, int right);

  // Assignment operator

  __unsignedbv& operator=(const __unsignedbv<W>& bv)
  {
    if(this == &bv) return *this;

    for(unsigned i = 0; i < UNSIGNEDBV_SIZEOF_DATA(data); i++)
      data_at(i) = bv.data_at(i);
    return *this;
  }

  // Logical assignment operators

  __unsignedbv& operator&=(const __unsignedbv<W>& bv)
  {
    if(this == &bv) return *this;

    for(unsigned i = 0; i < UNSIGNEDBV_SIZEOF_DATA(data); i++)
      data_at(i) &= bv.data_at(i);
    return *this;
  }

  __unsignedbv& operator|=(const __unsignedbv<W>& bv)
  {
    if(this == &bv) return *this;

    for(unsigned i = 0; i < UNSIGNEDBV_SIZEOF_DATA(data); i++)
      data_at(i) |= bv.data_at(i);
    return *this;
  }

  __unsignedbv& operator^=(const __unsignedbv<W>& bv)
  {
    if(this == &bv) return *this;

    for(unsigned i = 0; i < UNSIGNEDBV_SIZEOF_DATA(data); i++)
      data_at(i) ^= bv.data_at(i);
    return *this;
  }

  __unsignedbv operator ~ () const
  {
    __unsignedbv tmp;

    for(unsigned i = 0; i < UNSIGNEDBV_SIZEOF_DATA(data); i++)
      tmp.data_at(i) = ~(data_at(i));
    return  tmp;
  }

  // Arithemtic assignment operators

  __unsignedbv& operator +=(const __unsignedbv&);
  __unsignedbv& operator -=(const __unsignedbv&);
  __unsignedbv& operator *=(const __unsignedbv&);
  __unsignedbv& operator /=(const __unsignedbv&);
  __unsignedbv& operator %=(const __unsignedbv&);

  // Shift assignment operators

  __unsignedbv&  operator <<= (int i);
  __unsignedbv&  operator >>= (int i);

  // Set range
  template <unsigned W2, unsigned R>
  void set_range(const __unsignedbv<W2>& b)
  {
#ifdef __BV_DEBUG
    assert(W2 + R <= W);
#endif

    const unsigned l = (W2-1+R)/UINT_BSIZE;
    const unsigned r = R/UINT_BSIZE;
    const unsigned l_rem = (W2-1+R)%UINT_BSIZE;
    const unsigned r_rem = R%UINT_BSIZE;

    if(l != r)
    {
      if(r_rem == 0)
      {      
        unsigned j = 0;
        for(unsigned i = r; i < l; i++,j++)
          data_at(i) = b.data_at(j);

#ifdef __BV_DEBUG
        assert(j < W2);
#endif

        if(l_rem+1 == UINT_BSIZE)
          data_at(l) = b.data_at(j);
        else
          data_at(l) =  b.data_at(j) >> W2%UINT_BSIZE | (data_at(l) &  (UINT_MAX << (r_rem+1)));
      }
      else
      {
        if((W2-1)/UINT_BSIZE+1 == l-r+1)
        {
          data_at(r) = (data_at(r) &  ~(UINT_MAX << r_rem)) | b.data_at(0) <<  r_rem;
          unsigned j = 1;
          for(unsigned i = r+1; i < l ; i++,j++)
            data_at(i) = (b.data_at(j-1) >> (UINT_BSIZE-r_rem))  | b.data_at(j) <<  r_rem;
          
         if(l_rem+1 == UINT_BSIZE)
         {
#ifdef __BV_DEBUG
            assert(l_rem == UINT_BSIZE-r_rem);
#endif
            data_at(l) =  (b.data_at(j-1) >> (UINT_BSIZE-r_rem)) | b.data_at(j) <<  r_rem;
         }
         else
         {
            data_at(l) =  (b.data_at(j-1) >> (UINT_BSIZE-r_rem)) | b.data_at(j) << r_rem |
              (data_at(l) &  (UINT_MAX << (l_rem+1)));
         }
        }
        else
        {
          data_at(r) = (data_at(r) &  ~(UINT_MAX << r_rem)) | b.data_at(0) <<  r_rem;
          unsigned j = 1;
          for(unsigned i = r+1; i < l ; i++,j++)
            data_at(i) = (b.data_at(j-1) >> (UINT_BSIZE-r_rem))  | b.data_at(j) <<  r_rem;
  
#ifdef __BV_DEBUG
          assert(l_rem+1 != UINT_BSIZE);
#endif
          data_at(l) =  (b.data_at(j-1) >> (UINT_BSIZE-r_rem)) | (data_at(l) &  (UINT_MAX << (l_rem+1)));
        }
      }
    }
    else
    {
      const unsigned r_mask = ~(UINT_MAX  << r_rem); // `1' means: keep bit
      if(l_rem == UINT_BSIZE-1)
      {
         data_at(r) = ((data_at(r) & r_mask) | (b.data_at(0) << r_rem));
      }
      else
      {
         const unsigned l_mask = UINT_MAX << l_rem + 1;  // `1' means: keep the bit
         const unsigned lr_mask = l_mask | r_mask;
         data_at(r) = ((data_at(r) & lr_mask) | (b.data_at(0) << r_rem));
      }
    }

#ifdef __BV_DEBUG
    for(unsigned i = 0; i < W2; i++){
      bool b1 =  (*this)[R+i]; bool b2 =  b[i];
      assert(b1 == b2);
    }
#endif

  }

  //Set bit
  void set_bit(unsigned i, bool b)
  {
#ifdef __BV_DEBUG
    assert(i < W);
#endif

    const unsigned index = i/UINT_BSIZE;
    const unsigned rem =  i%UINT_BSIZE;
    
    if(b)
      data_at(index) =  ((1 << rem) | data_at(index)) ;
    else
      data_at(index) = ((~(1 << rem)) & data_at(index));
  }

  // Conversion functions

  bool to_bool() const
  {
    for(unsigned i = 0; i < UNSIGNEDBV_SIZEOF_DATA(data) ; i++)
      if(this->data_at(i))
        return true;
    return false;
  }

  int to_int() const
  {
    return data[0];
  }

  unsigned to_unsigned() const
  {
    return data[0];
  }

  // Bit extraction

  template <int W2>
    bool operator[] (const __unsignedbv<W2> & bv) const
    {
      return (*this)[bv.to_int()];
    }

  template <int W2>
    bool operator[] (const __signedbv<W2> & bv) const
    {
      return (*this)[bv.to_int()];
    }

  bool operator[](int i) const
  {
#ifdef __BV_DEBUG
    assert(0 <= i &&i < W);
#endif

    int x = i/UINT_BSIZE;
    int y = i%UINT_BSIZE;
    int mask = 1 << y;
    return (data_at(x) & mask);
  }

  unsigned int& data_at(unsigned  v)
  {
#ifdef __BV_DEBUG
    assert(0 <= v && v < (W/UINT_BSIZE)+1);
#endif
    return data[v];
  }

  unsigned int data_at(unsigned  v) const
  {
#ifdef __BV_DEBUG
    assert(v < (W/UINT_BSIZE)+1);
#endif
    return data[v];
  }
};

// Conversion constructor for type signedbv

template<int W> template<int W2>
  __INLINE
__unsignedbv<W>::__unsignedbv(const __unsignedbv<W2>& bv)
{
  if(W < W2)
  {
    for(unsigned i = 0; i < UNSIGNEDBV_SIZEOF_DATA(data); i++)
      data_at(i) = bv.data_at(i);

    if(W%UINT_BSIZE != 0)
    {
      data_at(UNSIGNEDBV_SIZEOF_DATA(data)-1) <<= UINT_BSIZE -  W%UINT_BSIZE;
      data_at(UNSIGNEDBV_SIZEOF_DATA(data)-1) >>= UINT_BSIZE -  W%UINT_BSIZE;
    }

  }
  else
  {
    for(unsigned i = 0; i < UNSIGNEDBV_SIZEOF_DATA(bv.data); i++)
      data_at(i) = bv.data_at(i);

    for(unsigned i = UNSIGNEDBV_SIZEOF_DATA(bv.data); i < UNSIGNEDBV_SIZEOF_DATA(data); i++)
      data_at(i) = 0;
  }
}


// Bit-extraction constructor

template<int W> template<int W2>
  __INLINE
__unsignedbv<W>::__unsignedbv(const __unsignedbv<W2>& bv, int left, int right)
{
#ifdef __BV_DEBUG
  assert( W == left - right + 1 && W2 > left && left > right && right >= 0 );
#endif

  // optimized case
  if( left/UINT_BSIZE == right/UINT_BSIZE)
  {
    data_at(0) = bv.data_at(right/UINT_BSIZE);
    data_at(0) <<= UINT_BSIZE -1 - (left%UINT_BSIZE);
    data_at(0) >>= (UINT_BSIZE -1 - (left%UINT_BSIZE)) + (right%UINT_BSIZE);
  }
  else
  {
    __unsignedbv<W2> tmp(bv);
    tmp <<= W2-1-left;
    tmp >>= (W2-1-left) + right;
    *this = (__unsignedbv<W>)tmp;
  }
}

// Comparison operators

template<int W>
  __INLINE
bool operator==(const __unsignedbv<W>& bv1, const __unsignedbv<W>& bv2)
{
  for(unsigned i = 0; i < UNSIGNEDBV_SIZEOF_DATA(bv1.data) ; i++)
    if (bv1.data_at(i) != bv2.data_at(i))return false;
  return true;
}

template<int W>
bool operator < (const __unsignedbv<W>& bv1, const __unsignedbv<W>& bv2);


#ifdef __LITTLE_ENDIAN__
#if (__SIZEOF_LONG_LONG__  == 2 * __SIZEOF_INT__)

template<>
  __INLINE
bool operator==(const __unsignedbv<LONG_LONG_UINT_BSIZE>& bv1, const __unsignedbv<LONG_LONG_UINT_BSIZE>& bv2)
{
  return (*(long long unsigned*) bv1.data) == (*(long long unsigned*) bv2.data);
}

template<>
  __INLINE
bool operator<(const __unsignedbv<LONG_LONG_UINT_BSIZE>& bv1, const __unsignedbv<LONG_LONG_UINT_BSIZE>& bv2)
{
  return (*(long long unsigned*) bv1.data) < (*(long long unsigned*) bv2.data);
}

#endif // __LITTLE_ENDIAN__
#endif // (__SIZEOF_LONG_LONG__  == 2 * __SIZEOF_INT__


template<int W>
  __INLINE
bool operator !=(const __unsignedbv<W>& bv1, const __unsignedbv<W>& bv2)
{
  return !(bv1 == bv2);
}


  template<>
bool operator < <8> (const __unsignedbv<8>& bv1, const __unsignedbv<8>& bv2)
{
  return bv1.data[0] < bv2.data[0];
}

  template<>
bool operator < <UINT_BSIZE> (const __unsignedbv<UINT_BSIZE>& bv1, const __unsignedbv<UINT_BSIZE>& bv2)
{
  return bv1.data[0] < bv2.data[0];
}


template<int W>
  __INLINE
bool operator > (const __unsignedbv<W>& bv1, const __unsignedbv<W>& bv2)
{
  return bv2 < bv1;
}

template<int W>
  __INLINE
bool operator <= (const __unsignedbv<W>& bv1, const __unsignedbv<W>& bv2)
{
  return !( bv1 > bv2);
}

template<int W>
  __INLINE
bool operator >= (const __unsignedbv<W>& bv1, const __unsignedbv<W>& bv2)
{
  return !( bv1 < bv2);
}

// logical operators

template<int W>
  __INLINE
__unsignedbv<W> operator & (const __unsignedbv<W>& bv1, const __unsignedbv<W>& bv2)
{
  __unsignedbv<W> tmp(bv1);
  tmp &= bv2;
  return  tmp;
}

template<int W>
  __INLINE
__unsignedbv<W> operator | (const __unsignedbv<W>& bv1, const __unsignedbv<W>& bv2)
{
  __unsignedbv<W> tmp(bv1);
  tmp |= bv2;
  return  tmp;
}

template<int W>
  __INLINE
__unsignedbv<W> operator ^ (const __unsignedbv<W>& bv1, const __unsignedbv<W>& bv2)
{
  __unsignedbv<W> tmp(bv1);
  tmp ^= bv2;
  return  tmp;
}

// arithmetic operators

template<int W>
  __INLINE
__unsignedbv<W>& __unsignedbv<W>::operator +=(const __unsignedbv<W>& bv)
{
  unsigned carry = 0;
  for(unsigned i = 0; i < UNSIGNEDBV_SIZEOF_DATA(data)-1 ; i++)
  {
    unsigned add_lw = carry + (data_at(i) >> 1) + (bv.data_at(i) >> 1);
    unsigned add_up = (data_at(i) >> (UINT_BSIZE-1)) + (bv.data_at(i) >> (UINT_BSIZE-1));
    data_at(i) += add_up;
    carry = add_up >> 1;  
#ifdef __BV_DEBUG
    assert(add_up < 4);
#endif
  }

  data_at(UNSIGNEDBV_SIZEOF_DATA(data)-1) += bv.data_at(UNSIGNEDBV_SIZEOF_DATA(data)-1);

  if(W%UINT_BSIZE !=0)
  {
    data_at(SIGNEDBV_SIZEOF_DATA(data)-1) <<= UINT_BSIZE-W%UINT_BSIZE;
    data_at(SIGNEDBV_SIZEOF_DATA(data)-1) >>= UINT_BSIZE-W%UINT_BSIZE;
  }
}

#ifdef __LITTLE_ENDIAN__
template<>
  __INLINE
__unsignedbv<8>& __unsignedbv<8>::operator +=(const __unsignedbv<8>& bv)
{
#ifdef __BV_DEBUG
  assert(this != &bv);
#endif

  unsigned char c = *((unsigned char*)data) + *((unsigned char*)bv.data);
  data[0] = c;
  return *this;
}
#endif

#ifdef __LITTLE_ENDIAN__
template<>
  __INLINE
__unsignedbv<8>& __unsignedbv<8>::operator -=(const __unsignedbv<8>& bv)
{
#ifdef __BV_DEBUG
  assert(this != &bv);
#endif

  unsigned char c = *((unsigned char*)data) - *((unsigned char*)bv.data);
  data[0] = c;
  return *this;
}
#endif


#ifdef __LITTLE_ENDIAN__
template<>
  __INLINE
__unsignedbv<8>& __unsignedbv<8>::operator *=(const __unsignedbv<8>& bv)
{
#ifdef __BV_DEBUG
  assert(this != &bv);
#endif

  unsigned char c = (*((unsigned char*)data)) *  (*((unsigned char*)bv.data));
  data[0] = c;
  return *this;
}
#endif

#ifdef __LITTLE_ENDIAN__
template<>
  __INLINE
__unsignedbv<8>& __unsignedbv<8>::operator /=(const __unsignedbv<8>& bv)
{
#ifdef __BV_DEBUG
  assert(this != &bv);
#endif

  unsigned char c = (*((unsigned char*)data)) /  (*((unsigned char*)bv.data));
  data[0] = c;
  return *this;
}
#endif

#ifdef __LITTLE_ENDIAN__
template<>
  __INLINE
__unsignedbv<8>& __unsignedbv<8>::operator %=(const __unsignedbv<8>& bv)
{
#ifdef __BV_DEBUG
  assert(this != &bv);
#endif

  unsigned char c = (*((unsigned char*)data)) %  (*((unsigned char*)bv.data));
  data[0] = c;
  return *this;
}
#endif

template<>
  __INLINE
__unsignedbv<UINT_BSIZE>& __unsignedbv<UINT_BSIZE>::operator +=(const __unsignedbv<UINT_BSIZE>& bv)
{
#ifdef __BV_DEBUG
  assert(this != &bv);
#endif

  data[0] += bv.data[0];
  return *this;
}

template<>
  __INLINE
__unsignedbv<UINT_BSIZE>& __unsignedbv<UINT_BSIZE>::operator -=(const __unsignedbv<UINT_BSIZE>& bv)
{
#ifdef __BV_DEBUG
  assert(this != &bv);
#endif

  data[0] -= bv.data[0];
  return *this;
}

template<>
  __INLINE
__unsignedbv<UINT_BSIZE>& __unsignedbv<UINT_BSIZE>::operator *=(const __unsignedbv<UINT_BSIZE>& bv)
{
#ifdef __BV_DEBUG
  assert(this != &bv);
#endif

  data[0] *= bv.data[0];
  return *this;
}

template<>
  __INLINE
__unsignedbv<UINT_BSIZE>& __unsignedbv<UINT_BSIZE>::operator /=(const __unsignedbv<UINT_BSIZE>& bv)
{
#ifdef __BV_DEBUG
  assert(this != &bv);
#endif

  data[0] /= bv.data[0];
  return *this;
}

template<>
  __INLINE
__unsignedbv<UINT_BSIZE>& __unsignedbv<UINT_BSIZE>::operator %=(const __unsignedbv<UINT_BSIZE>& bv)
{
#ifdef __BV_DEBUG
  assert(this != &bv);
#endif

  data[0] %= bv.data[0];
  return *this;
}

#ifdef __LITTLE_ENDIAN__
#if (__SIZEOF_LONG_LONG__  == 2 * __SIZEOF_INT__)

template<>
  __INLINE
__unsignedbv<LONG_LONG_UINT_BSIZE>& __unsignedbv<LONG_LONG_UINT_BSIZE>::operator +=(const __unsignedbv<LONG_LONG_UINT_BSIZE>& bv)
{
#ifdef __BV_DEBUG
  assert(this != &bv);
#endif

  (*(long long unsigned*)data) -= (*(long long unsigned*)bv.data) ;
  return *this;
}

template<>
  __INLINE
__unsignedbv<LONG_LONG_UINT_BSIZE>& __unsignedbv<LONG_LONG_UINT_BSIZE>::operator -=(const __unsignedbv<LONG_LONG_UINT_BSIZE>& bv)
{
#ifdef __BV_DEBUG
  assert(this != &bv);
#endif

  (*(long long unsigned*)data) -= (*(long long unsigned*)bv.data) ;
  return *this;
}

template<>
  __INLINE
__unsignedbv<LONG_LONG_UINT_BSIZE>& __unsignedbv<LONG_LONG_UINT_BSIZE>::operator *=(const __unsignedbv<LONG_LONG_UINT_BSIZE>& bv)
{
#ifdef __BV_DEBUG
  assert(this != &bv);
#endif

  (*(long long unsigned*)data) *= (*(long long unsigned*)bv.data) ;
  return *this;
}

template<>
  __INLINE
__unsignedbv<LONG_LONG_UINT_BSIZE>& __unsignedbv<LONG_LONG_UINT_BSIZE>::operator /=(const __unsignedbv<LONG_LONG_UINT_BSIZE>& bv)
{
#ifdef __BV_DEBUG
  assert(this != &bv);
#endif

  (*(long long unsigned*)data) /= (*(long long unsigned*)bv.data) ;
  return *this;
}

template<>
  __INLINE
__unsignedbv<LONG_LONG_UINT_BSIZE>& __unsignedbv<LONG_LONG_UINT_BSIZE>::operator %=(const __unsignedbv<LONG_LONG_UINT_BSIZE>& bv)
{
#ifdef __BV_DEBUG
  assert(this != &bv);
#endif

  (*(long long unsigned*)data) %= (*(long long unsigned*)bv.data) ;
  return *this;
}

#endif // __LITTLE_ENDIAN__
#endif // (__SIZEOF_LONG_LONG__  == 2 * __SIZEOF_INT__

template<int W>
  __INLINE
__unsignedbv<W> operator +(const __unsignedbv<W>& bv1, const __unsignedbv<W>& bv2)
{
  __unsignedbv<W> tmp(bv1);
  tmp += bv2;
  return  tmp;
}

template<class T>
  __INLINE
T* operator +(T* pt, const __unsignedbv<32>& bv)
{
  return  pt + bv.to_unsigned();
}

template<int W>
  __INLINE
__unsignedbv<W> operator -(const __unsignedbv<W>& bv1, const __unsignedbv<W>& bv2)
{
  __unsignedbv<W> tmp(bv1);
  tmp -= bv2;
  return  tmp;
}

template<int W>
  __INLINE
__unsignedbv<W> operator *(const __unsignedbv<W>& bv1, const __unsignedbv<W>& bv2)
{
  __unsignedbv<W> tmp(bv1);
  tmp *= bv2;
  return  tmp;
}

template<int W>
  __INLINE
__unsignedbv<W> operator /(const __unsignedbv<W>& bv1, const __unsignedbv<W>& bv2)
{
  __unsignedbv<W> tmp(bv1);
  tmp /= bv2;
  return  tmp;
}

template<int W>
  __INLINE
__unsignedbv<W> operator %(const __unsignedbv<W>& bv1, const __unsignedbv<W>& bv2)
{
  __unsignedbv<W> tmp(bv1);
  tmp %= bv2;
  return  tmp;
}

// Shift assignment operators

template<>
  __INLINE
__unsignedbv<8>&  __unsignedbv<8>::operator <<= (int i)
{
  data[0] = (unsigned char)(data[0] << i);
  return *this;
}

template<>
  __INLINE
__unsignedbv<UINT_BSIZE>&  __unsignedbv<UINT_BSIZE>::operator <<= (int i)
{
  data[0] <<= i;
  return *this;
}

template<>
  __INLINE
__unsignedbv<8>&  __unsignedbv<8>::operator >>= (int i)
{
  data[0] = (unsigned char)(data[0] >> i);
  return *this;
}

template<>
  __INLINE
__unsignedbv<UINT_BSIZE>&  __unsignedbv<UINT_BSIZE>::operator >>= (int i)
{
  data[0] >>= i;
  return *this;
}

  template<int W>
__unsignedbv<W>&  __unsignedbv<W>::operator <<= (int shift)
{
  for(unsigned j = 0; j < shift/UINT_BSIZE; j++)
  {
    for(int i = UNSIGNEDBV_SIZEOF_DATA(data)-1 ; i > 0 ; i--)
    {
      data_at(i) = data_at(i-1);
    }
    data_at(0) = 0;
  }

  int small_shift = shift%UINT_BSIZE;
  if(shift%UINT_BSIZE != 0)
  {
    for(int i = UNSIGNEDBV_SIZEOF_DATA(data)-1 ; i > 0 ; i--)
    {
      data_at(i) <<= small_shift;
      data_at(i) += data_at(i-1) >> (UINT_BSIZE - small_shift);

    }
    data_at(0) <<= shift%UINT_BSIZE;
  }

  if(W%UINT_BSIZE !=0)
  {
    data_at(SIGNEDBV_SIZEOF_DATA(data)-1) <<= UINT_BSIZE-W%UINT_BSIZE;
    data_at(SIGNEDBV_SIZEOF_DATA(data)-1) >>= UINT_BSIZE-W%UINT_BSIZE;
  }

  return *this;
}

  template<int W>
__unsignedbv<W>&  __unsignedbv<W>::operator >>= (int shift)
{
  for(unsigned j = 0; j < shift/UINT_BSIZE; j++)
  {
    for(unsigned i = 0; i < UNSIGNEDBV_SIZEOF_DATA(data)-1; i++)
    {
      data_at(i) = data_at(i+1);
    }
    data_at(UNSIGNEDBV_SIZEOF_DATA(data)-1) = 0;
  }

  int small_shift = shift%UINT_BSIZE;

  if(small_shift != 0)
  {
    for(unsigned i = 0; i < UNSIGNEDBV_SIZEOF_DATA(data)-1; i++)
    {
      data_at(i) = data_at(i) >> small_shift;
      data_at(i) += data_at(i+1) << (UINT_BSIZE - small_shift);
    }
    data_at(UNSIGNEDBV_SIZEOF_DATA(data)-1) >>= small_shift;
  }
  return *this;
}


#ifdef __LITTLE_ENDIAN__
#if (__SIZEOF_LONG_LONG__  == 2 * __SIZEOF_INT__)

template<>
  __INLINE
__unsignedbv<LONG_LONG_UINT_BSIZE>&  __unsignedbv<LONG_LONG_UINT_BSIZE>::operator <<= (int i)
{
  *((long long unsigned*) data) <<= i;
  return *this;
}

template<>
  __INLINE
__unsignedbv<LONG_LONG_UINT_BSIZE>&  __unsignedbv<LONG_LONG_UINT_BSIZE>::operator >>= (int i)
{
  *((long long unsigned*) data) >>= i;
  return *this;
}
#endif // __LITTLE_ENDIAN__
#endif // (__SIZEOF_LONG_LONG__  == 2 * __SIZEOF_INT__

// Shift operator

template<int W>
  __INLINE
__unsignedbv<W> operator << (const __unsignedbv<W>& bv, int i)
{
  __unsignedbv<W> tmp(bv);
  tmp <<= i;
  return tmp;
}

template<int W>
  __INLINE
__unsignedbv<W> operator >> (const __unsignedbv<W>& bv, int i)
{
  __unsignedbv<W> tmp(bv);
  tmp >>= i;
  return tmp;
}


// Printing operator

#ifdef __LITTLE_ENDIAN__
  template<int W>
std::ostream& operator << ( std::ostream& os, const __unsignedbv<W>& bv)
{
  for(int i = UNSIGNEDBV_SIZEOF_DATA(bv.data) -1 ; i >= 0 ; i--)
  {
    os << std:: hex << bv.data_at(i) << std::dec;
    os <<'\'' ;
  }
  return os;
}
#endif

// Concatenation

template<int W1, int W2>
  __INLINE
__unsignedbv<W1+W2> __concatenation(const __unsignedbv<W1>& bv1, const __unsignedbv<W2>& bv2)
{
  __unsignedbv<W1+W2> tmp(bv1);
  tmp <<= W2;
  tmp |= (__unsignedbv<W1+W2>)bv2;

  return tmp;
}

#endif
