/*******************************************************************\
 *
 *
 * Author: Blanc Nicolas
 *
 * 
\*******************************************************************/


#ifndef SC_FIFO_H
#define SC_FIFO_H

#include <scoot_tags.h>

#include <sc_event.h>

namespace sc_core {

  class sc_fifo_base: public scoot_channel
  {
    public:
      sc_fifo_base(){}

      // Events checking

      const sc_event& data_read_event() const
      {
        return _data_read;
      }
      const sc_event& data_written_event() const
      {
        return _data_written;
      }

    protected:
      // Disabled
      sc_fifo_base(const sc_fifo_base&);

      sc_event _data_read;
      sc_event _data_written;
  };

  template <class T>
    class sc_fifo: public sc_fifo_base
  {
    public:

      // Constructors

      explicit sc_fifo( int size = 16 ):
        capacity(size), offset(0), size_rd(0)
    {
      data = new T[size];
    }

      explicit sc_fifo( const char* name, int size = 16):
        capacity(size), offset(0), size_rd(0)
#ifdef SCOOT_SKIP_NOTIFICATION_PHASES
        , data_read(false), data_written(false)
#endif
        {
          data = new T[size];
        }

      // Destructors

      ~sc_fifo();

      // Read access

      T read()
      {
        while(size_rd == offset)
        {
          wait(data_written_event());
        }

        T t = data[size_rd - 1];

        size_rd--;

#ifdef SCOOT_SKIP_NOTIFICATION_PHASES
        notify_read = true;
#else
        _data_read.notify(0);
#endif

        return t;
      }

      void read( T& t)
      {
        t = read();
      }

      bool nb_read( T& t)
      {
        if(size_rd == offset)
        {
          return false;
        }

        t = data[size_rd - 1];
        size_rd--;

#ifdef SCOOT_SKIP_NOTIFICATION_PHASES
        notify_read = true;
#else
        _data_read.notify(0);
#endif

        return true;
      }

      operator T ()
      {
        return read();
      }

      // Write access

      void write( const T& t)
      {
        while(size_rd == capacity)
        {
          wait(data_read_event());
        }

        for(int i = size_rd; i >= 0; i--)
        {
          data[i] = data[i-1];
        }

        offset++;
        size_rd++;

        data[0] = t;

#ifdef SCOOT_SKIP_NOTIFICATION_PHASES
        notify_written = true;
#else
        _data_written.notify(0);
#endif
      }

      bool nb_write( const T& t)
      {
        if(size_rd == capacity)
        {
          return false;
        }

        for(int i = size_rd; i >= 0; i--)
        {
          data[i] = data[i-1];
        }

        offset++;
        size_rd++;

        data[0] = t;

#ifdef SCOOT_SKIP_NOTIFICATION_PHASES
        notify_written = true;
#else
        _data_written.notify(0);
#endif

        return true;
      }

      sc_fifo<T>& operator= ( const T& t)
      {
        write(t);
        return *this;
      }

      int num_available() const
      {
        return size_rd - offset;
      }

      int num_free() const
      {
        return capacity - offset;
      }

    private:

      // Update

      void update()
      {
        offset = 0;

#ifdef SCOOT_SKIP_NOTIFICATION_PHASES
        _data_read.do_notify(notify_read);
        _data_written.do_notify(notify_written);
        notify_read = false;
        notify_written = false;
#endif
      }

      // Disabled
      sc_fifo( const sc_fifo<T>& );
      sc_fifo& operator= ( const sc_fifo<T>& );

      T* data;
      const int capacity;
      int offset;
      int size_rd;

#ifdef SCOOT_SKIP_NOTIFICATION_PHASES
      bool notify_read;
      bool notify_written;
#endif
  };

}

#endif
