/*******************************************************************\
 *
 *
 * Author: Blanc Nicolas
 *
 * 
\*******************************************************************/


#ifndef SC_SIGNAL_H
#define SC_SIGNAL_H

#include <scoot_tags.h>

#include <sc_event.h>

namespace sc_core {

  class sc_signal_base: public scoot_channel
  {
    public:
      sc_signal_base(){}

      const sc_event& value_changed_event() const
      {
        return _value_changed_event;
      }

    protected:
      // disabled
      sc_signal_base(const sc_signal_base&);

      sc_event _value_changed_event;
  };

  template <class T>
    class sc_signal: public sc_signal_base
  {
    public:

      // Constructors

      sc_signal():
        update_request(false),
        current_value((T)0), next_value((T)0){}

      explicit sc_signal(const char*):
        update_request(false),
        current_value((T)0), next_value((T)0){}

      // Access methods

      T read() const {
        return current_value;
      }

      void write(T t) {
        update_request = ( t!=current_value);
        next_value =  t;
      }

      // syntaxic sugar

      operator T () const
      { return read(); }

      sc_signal<T>& operator = ( const T& a )
      { write( a ); return *this; }

      sc_signal<T>& operator = ( const sc_signal<T>& a )
      { write( a.read() ); return *this; }

      // Event checking

      bool event() const {
        return _value_changed_event.notified();
      }

    private:

      void update()
      {
#ifdef SCOOT_SKIP_NOTIFICATION_PHASES
        if(update_request)
          current_value = next_value;
        _value_changed_event.do_notify(update_request);
        update_request = false;
#else
        if(update_request)
        {
          _value_changed_event.notify(0);
          current_value = next_value;
        }
        update_request = false;
#endif
      }

      bool update_request;
      T current_value;
      T next_value;

      // Disabled

      sc_signal(const sc_signal<T>&);
  };

  template <>
    class sc_signal<bool>: public sc_signal_base
    {
      public:

        // Constructors

        sc_signal():
          previous_value(false), current_value(false), next_value(false) {}

        explicit sc_signal(const char*):
          previous_value(false), current_value(false),next_value(false) {}

        // Access methods

        bool read() const {
          return current_value;
        }

        void write(bool v) {
          next_value = v;
        }

        // syntaxic sugar

        operator bool () const
        { return read(); }

        sc_signal<bool>& operator = ( bool a )
        { write( a ); return *this; }

        sc_signal<bool>& operator = ( const sc_signal<bool>& a )
        { write( a.read() ); return *this; }


        // Events

        const sc_event& posedge_event() const
        {
          return _posedge_event;
        }

        const sc_event& negedge_event() const
        {
          return _negedge_event;
        }

        // Event checking

        bool event() const {
          return previous_value != current_value;
        }

        bool posedge() const {
          return previous_value == false && current_value == true;
        }

        bool negedge() const {
          return previous_value == true && current_value == false;
        }

      protected:

        bool previous_value;
        bool current_value;
        bool next_value;

        sc_event _posedge_event;
        sc_event _negedge_event;

        void update()
        {
          previous_value = current_value;
          current_value = next_value;

#ifdef SCOOT_SKIP_NOTIFICATION_PHASES

          _value_changed_event.do_notify(event());
          _posedge_event.do_notify(posedge());
          _negedge_event.do_notify(negedge());

#else
          if(previous_value != current_value)
          {
            _value_changed_event.notify(0);

            if(current_value == true)
              _posedge_event.notify(0);
            else
              _negedge_event.notify(0);
          }
#endif
        }

      private:
        // Disabled

        sc_signal(const sc_signal<bool>&);
    };

#if 0
  template <>
    class sc_signal<sc_logic>: public sc_signal_base
    {
      public:

        // Constructors

        sc_signal();
        explicit sc_signal(const char*);

        // Access methods

        sc_logic read() const {
          return current_value;
        }

        void write(sc_logic v) {
          next_value = v;
        }

        // syntaxic sugar

        operator sc_logic () const
        { return read(); }

        sc_signal<sc_logic>& operator = ( sc_logic a )
        { write( a ); return *this; }

        sc_signal<sc_logic>& operator = ( const sc_signal<sc_logic>& a )
        { write( a.read() ); return *this; }

        // Event checking

        bool event()const {
          return previous_value != current_value;
        }

        bool posedge() const {
          return current_value == SC_LOGIC_1 && current_value != previous_value;
        }
        bool negedge() const {
          return current_value == SC_LOGIC_0 && current_value != previous_value;
        }

      private:

        void update() {
          previous_value = current_value;
          current_value = next_value;
        }

        sc_logic previous_value;
        sc_logic current_value;
        sc_logic next_value;

        // Disabled

        sc_signal(const sc_signal<sc_logic>&);
    };


  typedef sc_signal<sc_logic> sc_signal_resolved;

  template <int W>
    class sc_signal_rv: public sc_signal<sc_lv<W> >
  {
    public:
      sc_signal_rv(){}

    private:

      // Disabled
      sc_signal_rv(const sc_signal_rv&);
  };
#endif


}

#endif
