/*******************************************************************\
 *
 *
 * Author: Blanc Nicolas
 *
 * 
\*******************************************************************/


#ifndef SC_TIME_H
#define SC_TIME_H

#include <scoot_tags.h>

namespace sc_core {


  /*
   * The time resolution is set to 0.5 ns.
   *
   * This deviates form the SystemC standard as the
   * default time resolution is 1 ps. Furthermore,
   * 0.5 ns is not a power of 10.
   *
   * We set the time resolution to 0.5 ns to generate
   * a clock signal with a period of 1 ns and a duty
   * cycle of 50%.
   * 
   */

  enum sc_time_unit {SC_FS = 0, SC_PS, SC_NS, SC_US, SC_MS, SC_SEC};

  class sc_time
  {
    public:

      sc_time():sim_time(0LL){};

      sc_time(long long time): sim_time(time){}

      sc_time(long long time , sc_time_unit unit)
      {
        switch(unit)
        {
          case SC_FS:
            {
              sim_time = 2*time/1000000LL;
              break;
            }
          case SC_PS:
            {
              sim_time = 2*time/1000LL;
              break;
            }
          case SC_NS:
            {
              sim_time = 2*time;
              break;
            }
          case SC_US:
            {
              sim_time = 2*1000LL * time;
              break;
            }
          case SC_MS:
            {
              sim_time = 2*1000000LL * time;
              break;
            }
          default:
            {
              // SC_SEC
              sim_time = 2*1000000000LL * time;
              break;
            }
        }
      }

      sc_time( const sc_time& time): sim_time(time.sim_time){}

      sc_time& operator= ( const sc_time& time)
      { sim_time = time.sim_time; }

      bool operator== ( const sc_time& time) const
      { return sim_time == time.sim_time; }

      bool operator!= ( const sc_time& time) const
      { return sim_time != time.sim_time; }

      bool operator< ( const sc_time& time) const
      { return sim_time < time.sim_time; }

      bool operator<= ( const sc_time& time) const
      { return sim_time <= time.sim_time; }

      bool operator> ( const sc_time& time) const
      { return sim_time > time.sim_time; }

      bool operator>= ( const sc_time&  time) const
      { return sim_time >= time.sim_time; }

      sc_time& operator+= ( const sc_time& time)
      {
        sim_time += time.sim_time;
        return *this;
      }

      sc_time& operator-= ( const sc_time& time)
      {
        sim_time -= time.sim_time;
        return *this;
      }

      sc_time& operator*= ( long long coef)
      {
        sim_time *= coef;
        return *this;
      }

      sc_time& operator/= ( long long coef)
      {
        sim_time *= coef;
        return *this;
      }

      long long value() const
      { return sim_time; }

    private:

      long long sim_time;

  };

  const sc_time operator+ ( const sc_time& t1, const sc_time& t2)
  {
    sc_time t(t1);
    t += t2;
    return t;
  }

  const sc_time operator- ( const sc_time& t1, const sc_time& t2)
  {
    sc_time t(t1);
    t -= t2;
    return t;
  }

  const sc_time operator* ( const sc_time& t1, long long c)
  {
    sc_time t(t1);
    t *= c;
    return t;
  }

  const sc_time operator* ( long long c, const sc_time& t1)
  {
    sc_time t(t1);
    t *= c;
    return t;
  }

  const sc_time operator/ ( const sc_time& t1, long long c)
  {
    sc_time t(t1);
    t /= c;
    return t;
  }

  const sc_time operator/ ( long long c, const sc_time& t1)
  {
    sc_time t(t1);
    t /= c;
    return t;
  }

  const sc_time& sc_get_time_resolution()
  {
    static sc_time t(500LL, SC_PS);
    return t;
  }

  sc_time sc_time_stamp()
  {
    return sc_time(scoot_simulation_time);
  }

}

#endif
