/*****************************************************************************

  The following code is derived, directly or indirectly, from the SystemC
  source code Copyright (c) 1996-2008 by all Contributors.
  All Rights reserved.

  The contents of this file are subject to the restrictions and limitations
  set forth in the SystemC Open Source License Version 3.0 (the "License");
  You may not use this file except in compliance with such restrictions and
  limitations. You may obtain instructions on how to receive a copy of the
  License at http://www.systemc.org/. Software distributed by Contributors
  under the License is distributed on an "AS IS" basis, WITHOUT WARRANTY OF
  ANY KIND, either express or implied. See the License for the specific
  language governing rights and limitations under the License.

 *****************************************************************************/

#ifndef __TLM_FIFO_H__
#define __TLM_FIFO_H__



#include <scoot_tags.h>
#include <sc_event.h>
#include <sc_wait.h>

#include <tlm_core_ifs.h>

namespace tlm {

template <typename T>
class tlm_fifo :
  public tlm_get_peek_if<T>,
  public tlm_put_if<T>,
  public scoot_channel
{
public:

    // constructors

    explicit tlm_fifo( int size_ = 16 ):capacity(16),offset(0),size_rd(0) {
      data = new T[capacity];
    }

    explicit tlm_fifo( const char* name_, int size_ = 16 ):capacity(16),offset(0),size_rd(0)  {
      data = new T[capacity];
    }

    // tlm get interface
    void get(T& t)
    {
        while(size_rd == offset)
        {
		      sc_core::wait(data_written_event());
        }

        t = data[size_rd - 1];

        size_rd--;

        #ifdef SCOOT_SKIP_NOTIFICATION_PHASES
        notify_read = true;
        #else
        _data_read.notify(0);
        #endif
    }

    T get()
    {
	      while(size_rd == offset)
        {
		      sc_core::wait(data_written_event());
        }

        T t = data[size_rd - 1];

        size_rd--;

        #ifdef SCOOT_SKIP_NOTIFICATION_PHASES
        notify_read = true;
        #else
        _data_read.notify(0);
        #endif
    
	      return t;
    }

    bool nb_get( T& t)
    {
        if(size_rd == offset)
        {
            return false;
        }

        t = data[size_rd - 1];
        size_rd--;

        #ifdef SCOOT_SKIP_NOTIFICATION_PHASES
        notify_read = true;
        #else
        _data_read.notify(0);
	      #endif

        return true;
    }

    bool nb_can_get() const
    {
	    return size_rd != offset;
    }

    // tlm peek interface

    void peek(T& t) const
    {
	      while(size_rd == offset)
        {
		      sc_core::wait(data_written_event());
        }

        t = data[size_rd - 1];
    }

    
    T peek() const
    {
	      while(size_rd == offset)
        {
		      sc_core::wait(data_written_event());
        }

        return data[size_rd - 1];
    }

    bool nb_peek(T& t)const
    {
	    if(size_rd == offset)
	    {
		    return false;
	    }

	    t =  data[size_rd - 1];
    }
	

    // tlm put interface 

    void put( const T& t)
    {
        while(size_rd == capacity)
        {
		      sc_core::wait(data_read_event());
        }

        for(int i = size_rd; i >= 0; i--)
        {
           data[i] = data[i-1];
        }

        offset++;
        size_rd++;

        data[0] = t;

        #ifdef SCOOT_SKIP_NOTIFICATION_PHASES
        notify_written = true;
        #else
        _data_written.notify(0);
        #endif
    }

    bool nb_put( const T& t)
    {
        if(size_rd == capacity)
        {
            return false;
        }

        for(int i = size_rd; i >= 0; i--)
        {
           data[i] = data[i-1];
        }

        offset++;
        size_rd++;

        data[0] = t;

        #ifdef SCOOT_SKIP_NOTIFICATION_PHASES
        notify_written = true;
        #else
        _data_written.notify(0);
        #endif

        return true;
    }

    bool nb_can_put() const
    {
	    return size_rd != capacity;
    }

    const sc_core::sc_event& data_read_event() const
    {
        return _data_read;
    }
    const sc_core::sc_event& data_written_event() const
    {
        return _data_written;
    }

    const sc_core::sc_event& ok_to_get() const
    {
        return _data_written;
    }

    const sc_core::sc_event& ok_to_put() const
    {
        return _data_written;
    }

    const sc_core::sc_event& ok_to_peek() const
    {
        return _data_written;
    }

    // utility function
    virtual const scoot_channel& get_channel() const
    {
      return *this;
    }

private:

    void update()
    {
        offset = 0;

        #ifdef SCOOT_SKIP_NOTIFICATION_PHASES
        _data_read.do_notify(notify_read);
        _data_written.do_notify(notify_written);
        notify_read = false;
        notify_written = false;
        #endif
    }


    T* data;
    const int capacity;
    int offset;
    int size_rd;

    sc_core::sc_event _data_read;
    sc_core::sc_event _data_written;

    #ifdef SCOOT_SKIP_NOTIFICATION_PHASES
    bool notify_read;
    bool notify_written;
    #endif


private:

    // disabled
    tlm_fifo( const tlm_fifo& );
    tlm_fifo& operator = ( const tlm_fifo& );

};

}


#endif

